{-----------------------------------------------------------------------------
This Software is placed into Publlic Domain and distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied.
The Initial Developer is Martin Waldenburg
(Martin.Waldenburg@T-Online.de).
-----------------------------------------------------------------------------}
unit mwDelphiLexer;

interface

uses
  SysUtils, Classes,
  mwLanguageElements, mwBaseLexer, mwSimpleLexer, mwDelphiLanguageElements;

type

  TmwDelphiRange = (
    drNormal,
    drAnsiDirective,
    drBorlandDirective,
    drAnsiComment,
    drAssemblerReference,
    drBorlandComment
    );

  TmwDelphiLexData = class(TmwLexData)
  public
    Range: TmwDelphiRange;
  end;

  TmwDelphiLexer = class(TmwSimpleLexer)
  protected
    DirectiveKeyList: TmwLexKeyList;
    procedure CreateKeyLists; override;
    procedure DestroyKeyLists; override;
    procedure InitializeKeyTables; override;
    procedure IdentifierHandler; override;
    procedure AsteriskHandler; override;
    procedure SharpHandler; override;
    procedure DollarHandler; override;
    procedure ApostropheHandler; override;
    procedure RoundOpenHandler; override;
    procedure PeriodHandler; override;
    procedure SlashHandler; override;
    procedure ColonHandler; override;
    procedure LessHandler; override;
    procedure GreaterHandler; override;
    procedure AtHandler; override;
    procedure CurlyOpenHandler; override;
    procedure CurlyCloseHandler; override;
    procedure NextAnsiComment; virtual;
    procedure NextAssemblerReference; virtual;
    procedure NextBorlandComment; virtual;
  public
    Range: TmwDelphiRange;
    procedure AssignData(Source: TmwDelphiLexData); reintroduce;
    procedure AssignToData(Dest: TmwDelphiLexData); reintroduce;
    procedure AnsiCommentHandler; virtual;
    procedure AnsiDirectiveHandler; virtual;
    procedure AssemblerReferenceHandler; virtual;
    procedure BorlandCommentHandler; virtual;
    procedure BorlandDirectiveHandler; virtual;
    procedure Next; override;
  end;

implementation

{ TmwDelphiLexer }

procedure TmwDelphiLexer.AnsiCommentHandler;
begin
  Id := leAnsiComment;
  while Run < TheEnd do
  begin
    case Buf[Run] of
      #10: Exit;

      #13:
        case (Run + 1 < TheEnd) and (Buf[Run + 1] = #10) of
          True: Exit;
          False: inc(Run);
        end;

      '*':
        case (Run + 1 < TheEnd) and (Buf[Run + 1] = ')') of
          True:
            begin
              inc(Run, 2);
              Range := drNormal;
              Exit;
            end;
          False: inc(Run);
        end;
    else
      case Run + UTF8Width[Buf[Run]] < TheEnd of
        True: inc(Run, UTF8Width[Buf[Run]]);
        False: inc(Run);
      end;
    end;
  end;
end;

procedure TmwDelphiLexer.AnsiDirectiveHandler;
begin
  Id := leAnsiDirective;
  while Run < TheEnd do
  begin
    case Buf[Run] of
      #10: Exit;

      #13:
        case (Run + 1 < TheEnd) and (Buf[Run + 1] = #10) of
          True: Exit;
          False: inc(Run);
        end;

      '*':
        case (Run + 1 < TheEnd) and (Buf[Run + 1] = ')') of
          True:
            begin
              inc(Run, 2);
              Range := drNormal;
              Exit;
            end;
          False: inc(Run);
        end;
    else
      case Run + UTF8Width[Buf[Run]] < TheEnd of
        True: inc(Run, UTF8Width[Buf[Run]]);
        False: inc(Run);
      end;
    end;
  end;
end;

procedure TmwDelphiLexer.ApostropheHandler;
begin
  inc(Run);
  Id := leUnterminatedStringConstant;
  while Run < TheEnd do
    case Buf[Run] of
      #10, #13: Break;
      #39:
        begin
          inc(Run);
          case Run < TheEnd of
            True:
              case Buf[Run] = #39 of
                True: inc(Run);
                False:
                  begin
                    Id := leStringConstant;
                    Break;
                  end;
              end;
            False:
              begin
                Id := leStringConstant;
                Break;
              end;
          end;
        end;
    else inc(Run)
    end;
end;

procedure TmwDelphiLexer.AssemblerReferenceHandler;
begin
  Id := leAssemblerReference;
  while Run < TheEnd do
  begin
    case Buf[Run] of
      #10: Exit;

      #13:
        case (Run + 1 < TheEnd) and (Buf[Run + 1] = #10) of
          True: Exit;
          False: inc(Run);
        end;

      '}':
        begin
          inc(Run);
          Range := drNormal;
          Exit;
        end;
    else
      case Run + UTF8Width[Buf[Run]] < TheEnd of
        True: inc(Run, UTF8Width[Buf[Run]]);
        False: inc(Run);
      end;
    end;
  end;
end;

procedure TmwDelphiLexer.AssignData(Source: TmwDelphiLexData);
begin
  if (Source <> nil) then
  begin
    ExId := Source.ExId;
    FileName := Source.FileName;
    Id := Source.Id;
    fLinePosition := Source.LinePosition;
    fLineNumber := Source.LineNumber;
    Buf := Source.Buf;
    Range := Source.Range;
    Run := Source.Run;
    Start := Source.Start;
    TheEnd := Source.TheEnd;
  end;
end;

procedure TmwDelphiLexer.AssignToData(Dest: TmwDelphiLexData);
begin
  if (Dest <> nil) then
  begin
    Dest.ExId := ExId;
    Dest.FileName := FileName;
    Dest.Id := Id;
    Dest.LinePosition := LinePosition;
    Dest.LineNumber := LineNumber;
    Dest.Buf := Buf;
    Dest.Range := Range;
    Dest.Run := Run;
    Dest.Start := Start;
    Dest.TheEnd := TheEnd;
  end;
end;

procedure TmwDelphiLexer.AsteriskHandler;
begin
  inc(Run);
  Id := leAsterisk;
  if Range in [drAnsiComment, drAnsiDirective] then
    if Run < TheEnd then
      if Buf[Run] = ')' then
      begin
        inc(Run);
        Id := leEndOfAnsiComment;
        Range := drNormal;
      end;
end;

procedure TmwDelphiLexer.AtHandler;
begin
  inc(Run);
  Id := leAt;
  if (Run < TheEnd) and (Buf[Run] = '@') then
  begin
    Id := leDoubleAt;
    inc(Run);
  end;
end;

procedure TmwDelphiLexer.BorlandCommentHandler;
begin
  Id := leBorlandComment;
  while Run < TheEnd do
  begin
    case Buf[Run] of
      #10: Exit;

      #13:
        case (Run + 1 < TheEnd) and (Buf[Run + 1] = #10) of
          True: Exit;
          False: inc(Run);
        end;

      '}':
        begin
          inc(Run);
          Range := drNormal;
          Exit;
        end;
    else
      case Run + UTF8Width[Buf[Run]] < TheEnd of
        True: inc(Run, UTF8Width[Buf[Run]]);
        False: inc(Run);
      end;
    end;
  end;
end;

procedure TmwDelphiLexer.BorlandDirectiveHandler;
begin
  Id := leBorlandDirective;
  while Run < TheEnd do
  begin
    case Buf[Run] of
      #10: Exit;

      #13:
        case (Run + 1 < TheEnd) and (Buf[Run + 1] = #10) of
          True: Exit;
          False: inc(Run);
        end;

      '}':
        begin
          inc(Run);
          Range := drNormal;
          Exit;
        end;
    else
      case Run + UTF8Width[Buf[Run]] < TheEnd of
        True: inc(Run, UTF8Width[Buf[Run]]);
        False: inc(Run);
      end;
    end;
  end;
end;

procedure TmwDelphiLexer.ColonHandler;
begin
  inc(Run);
  Id := leColon;
  if (Run < TheEnd) and (Buf[Run] = '=') then
  begin
    Id := leAssignment;
    inc(Run);
  end;
end;

procedure TmwDelphiLexer.CreateKeyLists;
begin
  KeyList := TmwLexKeyList.Create(Self);
  DirectiveKeyList := TmwLexKeyList.Create(Self);
end;

procedure TmwDelphiLexer.CurlyCloseHandler;
begin
  inc(Run);
  Id := leCurlyClose;
  if Range in [drBorlandComment, drBorlandDirective] then
    Range := drNormal;
end;

procedure TmwDelphiLexer.CurlyOpenHandler;
begin
  inc(Run);
  Id := leCurlyOpen;
  if Range = drNormal then
    case Run < TheEnd of
      True:
        case Buf[Run] of
          '$':
            begin
              inc(Run);
              Id := leBeginOfBorlandDirective;
              Range := drBorlandDirective;
            end;
          '%':
            begin
              inc(Run);
              Id := leAssemblerReference;
              Range := drAssemblerReference;
              AssemblerReferenceHandler;
            end;
        else
          begin
            Range := drBorlandComment;
            BorlandCommentHandler;
          end;
        end;
      False:
        begin
          Range := drBorlandComment;
          BorlandCommentHandler;
        end;
    end;
end;

procedure TmwDelphiLexer.DestroyKeyLists;
begin
  KeyList.Free;
  DirectiveKeyList.Free;
end;

procedure TmwDelphiLexer.DollarHandler;
begin
  inc(Run);
  Id := leDollar;
  if (Run < TheEnd) and
    ((Buf[Run] >= '0') and (Buf[Run] <= '9')) or
    ((Buf[Run] >= 'a') and (Buf[Run] <= 'f')) or
    ((Buf[Run] >= 'A') and (Buf[Run] <= 'F')) then
  begin
    inc(Run);
    Id := leHexNumber;
    while (Run < TheEnd) and
      ((Buf[Run] >= '0') and (Buf[Run] <= '9')) or
      ((Buf[Run] >= 'a') and (Buf[Run] <= 'f')) or
      ((Buf[Run] >= 'A') and (Buf[Run] <= 'F')) do
      inc(Run);
  end;
end;

procedure TmwDelphiLexer.GreaterHandler;
begin
  inc(Run);
  Id := leGreater;
  if (Run < TheEnd) and (Buf[Run] = '=') then
  begin
    Id := leGreaterOrEqual;
    inc(Run);
  end;
end;

procedure TmwDelphiLexer.IdentifierHandler;
begin
  inc(Run);
  Id := leIdentifier;
  ExId := leIdentifier;
  while (Run < TheEnd) do
    case InIdentifiers[Buf[Run]] of
      True: inc(Run);
      False: Break;
    end;
  case InInternationalIdentifiers[Buf[Run]] of
    True: InternationalIdentifierHandler;
    False:
      case (Run + 1 < TheEnd) and (Buf[Run] > #191) of
        True: UTF8IdentifierHandler;
      else
        case Range of
          drNormal: KeyList.Hash(Start);
          drAnsiDirective: DirectiveKeyList.Hash(Start);
          drBorlandDirective: DirectiveKeyList.Hash(Start);
        end;
      end;
  end;
end;

procedure TmwDelphiLexer.InitializeKeyTables;
begin
  KeyList.Insensitive := True;
  DirectiveKeyList.Insensitive := True;

  {KeyWords}
  KeyList.Add('And', leAnd, leDelphiKeyWord);
  KeyList.Add('Array', leArray, leDelphiKeyWord);
  KeyList.Add('As', leAs, leDelphiKeyWord);
  KeyList.Add('Asm', leAsm, leDelphiKeyWord);
  KeyList.Add('Begin', leBegin, leDelphiKeyWord);
  KeyList.Add('Case', leCase, leDelphiKeyWord);
  KeyList.Add('Class', leClass, leDelphiKeyWord);
  KeyList.Add('Const', leConst, leDelphiKeyWord);
  KeyList.Add('Constructor', leConstructor, leDelphiKeyWord);
  KeyList.Add('Destructor', leDestructor, leDelphiKeyWord);
  KeyList.Add('Dispinterface', leDispinterface, leDelphiKeyWord);
  KeyList.Add('Div', leDiv, leDelphiKeyWord);
  KeyList.Add('Do', leDo, leDelphiKeyWord);
  KeyList.Add('Downto', leDownto, leDelphiKeyWord);
  KeyList.Add('Else', leElse, leDelphiKeyWord);
  KeyList.Add('End', leEnd, leDelphiKeyWord);
  KeyList.Add('Except', leExcept, leDelphiKeyWord);
  KeyList.Add('Exports', leExports, leDelphiKeyWord);
  KeyList.Add('File', leFile, leDelphiKeyWord);
  KeyList.Add('Final', leFinal, leDelphiKeyWord);
  KeyList.Add('Finalization', leFinalization, leDelphiKeyWord);
  KeyList.Add('Finally', leFinally, leDelphiKeyWord);
  KeyList.Add('For', leFor, leDelphiKeyWord);
  KeyList.Add('Function', leFunction, leDelphiKeyWord);
  KeyList.Add('Goto', leGoto, leDelphiKeyWord);
  KeyList.Add('If', leIf, leDelphiKeyWord);
  KeyList.Add('Implementation', leImplementation, leDelphiKeyWord);
  KeyList.Add('In', leIn, leDelphiKeyWord);
  KeyList.Add('Inherited', leInherited, leDelphiKeyWord);
  KeyList.Add('Initialization', leInitialization, leDelphiKeyWord);
  KeyList.Add('Inline', leInline, leDelphiKeyWord);
  KeyList.Add('Interface', leInterface, leDelphiKeyWord);
  KeyList.Add('Is', leIs, leDelphiKeyWord);
  KeyList.Add('Label', leLabel, leDelphiKeyWord);
  KeyList.Add('Library', leLibrary, leDelphiDirective);
  KeyList.Add('Mod', leMod, leDelphiKeyWord);
  KeyList.Add('Nil', leNil, leDelphiKeyWord);
  KeyList.Add('Not', leNot, leDelphiKeyWord);
  KeyList.Add('Object', leObject, leDelphiKeyWord);
  KeyList.Add('Of', leOf, leDelphiKeyWord);
  KeyList.Add('Or', leOr, leDelphiKeyWord);
  KeyList.Add('Packed', lePacked, leDelphiKeyWord);
  KeyList.Add('Procedure', leProcedure, leDelphiKeyWord);
  KeyList.Add('Program', leProgram, leDelphiKeyWord);
  KeyList.Add('Property', leProperty, leDelphiKeyWord);
  KeyList.Add('Raise', leRaise, leDelphiKeyWord);
  KeyList.Add('Record', leRecord, leDelphiKeyWord);
  KeyList.Add('Repeat', leRepeat, leDelphiKeyWord);
  KeyList.Add('Resourcestring', leResourcestring, leDelphiKeyWord);
  KeyList.Add('Sealed', leSealed, leDelphiKeyWord);
  KeyList.Add('Set', leSet, leDelphiKeyWord);
  KeyList.Add('Shl', leShl, leDelphiKeyWord);
  KeyList.Add('Shr', leShr, leDelphiKeyWord);
  KeyList.Add('String', leString, leDelphiKeyWord);
  KeyList.Add('Then', leThen, leDelphiKeyWord);
  KeyList.Add('Threadvar', leThreadvar, leDelphiKeyWord);
  KeyList.Add('To', leTo, leDelphiKeyWord);
  KeyList.Add('Try', leTry, leDelphiKeyWord);
  KeyList.Add('Type', leType, leDelphiKeyWord);
  KeyList.Add('Unit', leUnit, leDelphiKeyWord);
  KeyList.Add('Until', leUntil, leDelphiKeyWord);
  KeyList.Add('Uses', leUses, leDelphiKeyWord);
  KeyList.Add('Var', leVar, leDelphiKeyWord);
  KeyList.Add('While', leWhile, leDelphiKeyWord);
  KeyList.Add('With', leWith, leDelphiKeyWord);
  KeyList.Add('Xor', leXor, leDelphiKeyWord);

  {Directives}
  KeyList.Add('Absolute', leIdentifier, leAbsolute);
  KeyList.Add('Abstract', leIdentifier, leAbstract);
  KeyList.Add('Assembler', leIdentifier, leAssembler);
  KeyList.Add('At', leIdentifier, leAt);
  KeyList.Add('Automated', leIdentifier, leAutomated);
  KeyList.Add('Cdecl', leIdentifier, leCdecl);
  KeyList.Add('Contains', leIdentifier, leContains);
  KeyList.Add('Default', leIdentifier, leDefault);
  KeyList.Add('Deprecated', leIdentifier, leDeprecated);
  KeyList.Add('Dispid', leIdentifier, leDispid);
  KeyList.Add('Dynamic', leIdentifier, leDynamic);
  KeyList.Add('Experimental', leIdentifier, leExperimental);
  KeyList.Add('Export', leIdentifier, leExport);
  KeyList.Add('External', leIdentifier, leExternal);
  KeyList.Add('Far', leIdentifier, leFar);
  KeyList.Add('Forward', leIdentifier, leForward);
  KeyList.Add('Helper', leIdentifier, leHelper);
  KeyList.Add('Implements', leIdentifier, leImplements);
  KeyList.Add('Index', leIdentifier, leIndex);
  KeyList.Add('Local', leIdentifier, leLocal);
  KeyList.Add('Message', leIdentifier, leMessage);
  KeyList.Add('Name', leIdentifier, leName);
  KeyList.Add('Near', leIdentifier, leNear);
  KeyList.Add('Nodefault', leIdentifier, leNodefault);
  KeyList.Add('On', leIdentifier, leOn);
  KeyList.Add('Out', leIdentifier, leOut);
  KeyList.Add('Overload', leIdentifier, leOverload);
  KeyList.Add('Override', leIdentifier, leOverride);
  KeyList.Add('Package', leIdentifier, lePackage);
  KeyList.Add('Pascal', leIdentifier, lePascal);
  KeyList.Add('Platform', leIdentifier, lePlatform);
  KeyList.Add('Private', leIdentifier, lePrivate);
  KeyList.Add('Protected', leIdentifier, leProtected);
  KeyList.Add('Public', leIdentifier, lePublic);
  KeyList.Add('Published', leIdentifier, lePublished);
  KeyList.Add('Read', leIdentifier, leRead);
  KeyList.Add('Readonly', leIdentifier, leReadonly);
  KeyList.Add('Register', leIdentifier, leRegister);
  KeyList.Add('Reintroduce', leIdentifier, leReintroduce);
  KeyList.Add('Requires', leIdentifier, leRequires);
  KeyList.Add('Resident', leIdentifier, leResident);
  KeyList.Add('Safecall', leIdentifier, leSafecall);
  KeyList.Add('Stdcall', leIdentifier, leStdcall);
  KeyList.Add('Stored', leIdentifier, leStored);
  KeyList.Add('Strict', leIdentifier, leStrict);
  KeyList.Add('Varargs', leIdentifier, leVarargs);
  KeyList.Add('Virtual', leIdentifier, leVirtual);
  KeyList.Add('Write', leIdentifier, leWrite);
  KeyList.Add('Writeonly', leIdentifier, leWriteonly);

  {Additional}
  DirectiveKeyList.Add('A', leA, leIdentifier);
  DirectiveKeyList.Add('B', leB, leIdentifier);
  DirectiveKeyList.Add('C', leC, leIdentifier);
  DirectiveKeyList.Add('D', leD, leIdentifier);
  DirectiveKeyList.Add('E', leE, leIdentifier);
  DirectiveKeyList.Add('G', leG, leIdentifier);
  DirectiveKeyList.Add('H', leH, leIdentifier);
  DirectiveKeyList.Add('I', leI, leIdentifier);
  DirectiveKeyList.Add('J', leJ, leIdentifier);
  DirectiveKeyList.Add('L', leL, leIdentifier);
  DirectiveKeyList.Add('M', leM, leIdentifier);
  DirectiveKeyList.Add('O', leO, leIdentifier);
  DirectiveKeyList.Add('P', leP, leIdentifier);
  DirectiveKeyList.Add('Q', leQ, leIdentifier);
  DirectiveKeyList.Add('R', leR, leIdentifier);
  DirectiveKeyList.Add('T', leT, leIdentifier);
  DirectiveKeyList.Add('U', leU, leIdentifier);
  DirectiveKeyList.Add('V', leV, leIdentifier);
  DirectiveKeyList.Add('W', leW, leIdentifier);
  DirectiveKeyList.Add('X', leX, leIdentifier);
  DirectiveKeyList.Add('Y', leY, leIdentifier);
  DirectiveKeyList.Add('Z', leZ, leIdentifier);
  DirectiveKeyList.Add('And', leAndDirective, leIdentifier);
  DirectiveKeyList.Add('Off', leOff, leIdentifier);
  DirectiveKeyList.Add('YD', leYD, leIdentifier);
  DirectiveKeyList.Add('On', leOnDirective, leIdentifier);
  DirectiveKeyList.Add('If', leIfDirective, leIdentifier);
  DirectiveKeyList.Add('IfDef', leIfDef, leIdentifier);
  DirectiveKeyList.Add('Or', leOrDirective, leIdentifier);
  DirectiveKeyList.Add('GUI', leGUI, leIdentifier);
  DirectiveKeyList.Add('EndIf', leEndIf, leIdentifier);
  DirectiveKeyList.Add('IfEnd', leIfEnd, leIdentifier);
  DirectiveKeyList.Add('Else', leElseDirective, leIdentifier);
  DirectiveKeyList.Add('Define', leDefine, leIdentifier);
  DirectiveKeyList.Add('Align', leAlign, leIdentifier);
  DirectiveKeyList.Add('IfNDef', leIfNDef, leIdentifier);
  DirectiveKeyList.Add('Link', leLink, leIdentifier);
  DirectiveKeyList.Add('Defined', leDefined, leIdentifier);
  DirectiveKeyList.Add('UnDef', leUnDef, leIdentifier);
  DirectiveKeyList.Add('Declared', leDeclared, leIdentifier);
  DirectiveKeyList.Add('ElseIf', leElseIf, leIdentifier);
  DirectiveKeyList.Add('ImageBase', leImageBase, leIdentifier);
  DirectiveKeyList.Add('IfOpt', leIfOpt, leIdentifier);
  DirectiveKeyList.Add('SoName', leSoName, leIdentifier);
  DirectiveKeyList.Add('Include', leInclude, leIdentifier);
  DirectiveKeyList.Add('Hints', leHints, leIdentifier);
  DirectiveKeyList.Add('NoDefine', leNoDefine, leIdentifier);
  DirectiveKeyList.Add('IOChecks', leIOChecks, leIdentifier);
  DirectiveKeyList.Add('DebugInfo', leDebugInfo, leIdentifier);
  DirectiveKeyList.Add('Console', leConsole, leIdentifier);
  DirectiveKeyList.Add('Booleval', leBooleval, leIdentifier);
  DirectiveKeyList.Add('HPPEmit', leHPPEmit, leIdentifier);
  DirectiveKeyList.Add('RangeChecks', leRangeChecks, leIdentifier);
  DirectiveKeyList.Add('NoInclude', leNoInclude, leIdentifier);
  DirectiveKeyList.Add('Apptype', leApptype, leIdentifier);
  DirectiveKeyList.Add('SafeDivide', leSafeDivide, leIdentifier);
  DirectiveKeyList.Add('Resource', leResource, leIdentifier);
  DirectiveKeyList.Add('Warn', leWarn, leIdentifier);
  DirectiveKeyList.Add('Warnings', leWarnings, leIdentifier);
  DirectiveKeyList.Add('TypeInfo', leTypeInfo, leIdentifier);
  DirectiveKeyList.Add('SetPEFlags', leSetPEFlags, leIdentifier);
  DirectiveKeyList.Add('SoPrefix', leSoPrefix, leIdentifier);
  DirectiveKeyList.Add('StackFrames', leStackFrames, leIdentifier);
  DirectiveKeyList.Add('RunOnly', leRunOnly, leIdentifier);
  DirectiveKeyList.Add('SoSuffix', leSoSuffix, leIdentifier);
  DirectiveKeyList.Add('ReferenceInfo', leReferenceInfo, leIdentifier);
  DirectiveKeyList.Add('DesignOnly', leDesignOnly, leIdentifier);
  DirectiveKeyList.Add('Extension', leExtension, leIdentifier);
  DirectiveKeyList.Add('ImportedData', leImportedData, leIdentifier);
  DirectiveKeyList.Add('Description', leDescription, leIdentifier);
  DirectiveKeyList.Add('SoVersion', leSoVersion, leIdentifier);
  DirectiveKeyList.Add('Assertions', leAssertions, leIdentifier);
  DirectiveKeyList.Add('ImplicitBuild', leImplicitBuild, leIdentifier);
  DirectiveKeyList.Add('TypedAddress', leTypedAddress, leIdentifier);
  DirectiveKeyList.Add('LocalSymbols', leLocalSymbols, leIdentifier);
  DirectiveKeyList.Add('MinEnumSize', leMinEnumSize, leIdentifier);
  DirectiveKeyList.Add('WeakPackageUnit', leWeakPackageUnit, leIdentifier);
  DirectiveKeyList.Add('DefinitionInfo', leDefinitionInfo, leIdentifier);
  DirectiveKeyList.Add('MinStackSize', leMinStackSize, leIdentifier);
  DirectiveKeyList.Add('ObjExportAll', leObjExportAll, leIdentifier);
  DirectiveKeyList.Add('MaxStackSize', leMaxStackSize, leIdentifier);
  DirectiveKeyList.Add('LongStrings', leLongStrings, leIdentifier);
  DirectiveKeyList.Add('DenyPackageUnit', leDenyPackageUnit, leIdentifier);
  DirectiveKeyList.Add('ExternalSym', leExternalSym, leIdentifier);
  DirectiveKeyList.Add('OpenStrings', leOpenStrings, leIdentifier);
  DirectiveKeyList.Add('SetPEOptFlags', leSetPEOptFlags, leIdentifier);
  DirectiveKeyList.Add('OverFlowChecks', leOverFlowChecks, leIdentifier);
  DirectiveKeyList.Add('WriteableConst', leWriteableConst, leIdentifier);
  DirectiveKeyList.Add('Optimization', leOptimization, leIdentifier);
  DirectiveKeyList.Add('VarStringChecks', leVarStringChecks, leIdentifier);
  DirectiveKeyList.Add('ExtendedSyntax', leExtendedSyntax, leIdentifier);
  DirectiveKeyList.Add('RealCompatibility', leRealCompatibility, leIdentifier);
  DirectiveKeyList.Add('ResourceReserve', leResourceReserve, leIdentifier);
  DirectiveKeyList.Add('Region', leRegion, leIdentifier);
  DirectiveKeyList.Add('EndRegion', leEndRegion, leIdentifier);
end;

procedure TmwDelphiLexer.LessHandler;
begin
  inc(Run);
  Id := leLess;
  if Run < TheEnd then
    case Buf[Run] of
      '=':
        begin
          Id := leLessOrEqual;
          inc(Run);
        end;
      '>':
        begin
          Id := leNotEqual;
          inc(Run);
        end;
    end;
end;

procedure TmwDelphiLexer.Next;
begin
  case Range of
    drNormal: inherited Next;
    drAnsiDirective: inherited Next;
    drBorlandDirective: inherited Next;
    drAssemblerReference: NextAssemblerReference;
    drAnsiComment: NextAnsiComment;
    drBorlandComment: NextBorlandComment;
  end;
end;

procedure TmwDelphiLexer.NextAnsiComment;
begin
  Start := Run;
  ExId := leUnknown;
  case Run < TheEnd of
    True:
      case Buf[Run] of
        #10: LFHandler;
        #13: CRHandler;
      else
        AnsiCommentHandler;
      end;
    False: Id := leAtEnd;
  end
end;

procedure TmwDelphiLexer.NextAssemblerReference;
begin
  Start := Run;
  ExId := leUnknown;
  case Run < TheEnd of
    True:
      case Buf[Run] of
        #10: LFHandler;
        #13: CRHandler;
      else
        AssemblerReferenceHandler;
      end;
    False: Id := leAtEnd;
  end
end;

procedure TmwDelphiLexer.NextBorlandComment;
begin
  Start := Run;
  ExId := leUnknown;
  case Run < TheEnd of
    True:
      case Buf[Run] of
        #10: LFHandler;
        #13: CRHandler;
      else
        BorlandCommentHandler;
      end;
    False: Id := leAtEnd;
  end
end;

procedure TmwDelphiLexer.PeriodHandler;
begin
  inc(Run);
  Id := lePeriod;
  if Run < TheEnd then
    case Buf[Run] of
      '.':
        begin
          inc(Run);
          Id := leDotDot;
        end;
      ')':
        begin
          inc(Run);
          Id := leSquareClose;
        end;
    end;
end;

procedure TmwDelphiLexer.RoundOpenHandler;
begin
  inc(Run);
  Id := leRoundOpen;
  if Range = drNormal then
    if Run < TheEnd then
      case Buf[Run] of
        '.':
          begin
            inc(Run);
            Id := leSquareOpen;
          end;
        '*':
          begin
            inc(Run);
            Id := leAnsiComment;
            if Run < TheEnd then
              case Buf[Run] = '$' of
                True:
                  begin
                    inc(Run);
                    Range := drAnsiDirective;
                    Id := leBeginOfAnsiDirective;
                  end;
                False:
                  begin
                    Range := drAnsiComment;
                    AnsiCommentHandler;
                  end;
              end;
          end;
      end;
end;

procedure TmwDelphiLexer.SharpHandler;
begin
  inc(Run);
  Id := leSharp;
  if Run < TheEnd then
    case Buf[Run] of
      '$':
        begin
          inc(Run);
          Id := leCharHexConstant;
          while (Run < TheEnd) and
            ((Buf[Run] >= '0') and (Buf[Run] <= '9')) or
            ((Buf[Run] >= 'a') and (Buf[Run] <= 'f')) or
            ((Buf[Run] >= 'A') and (Buf[Run] <= 'F')) do
            inc(Run);
        end;
      '0'..'9':
        begin
          inc(Run);
          Id := leCharConstant;
          while (Run < TheEnd) and
            ((Buf[Run] >= '0') and (Buf[Run] <= '9')) do
            inc(Run);
        end;
    end;
end;

procedure TmwDelphiLexer.SlashHandler;
begin
  inc(Run);
  Id := leSlash;
  if Range = drNormal then
    case Run < TheEnd of
      True:
        if Buf[Run] = '/' then
        begin
          inc(Run);
          Id := leSlashComment;
          while Run < TheEnd do
            case Buf[Run] of
              #10: Break;
              #13: case (Run < TheEnd) and (Buf[Run + 1] = #10) of
                  True: Break;
                  False: inc(Run);
                end;
            else
              inc(Run);
            end;
        end;
    end;
end;

end.

